/*

 Chunk Expression Library for C++
 (c) 2004 Jonathan Bettencourt / Kreative Korporation

 This is pretty self-explanatory. For help on how to use this or what it does,
 see the man page for the cxl command line utility and/or the section on chunk
 expressions in the HyperCard Script Language Guide.


 This code is under the MIT license.

 Permission is hereby granted, free of charge, to any person obtaining a copy of
 this software and associated documentation files (the "Software"), to deal in the
 Software without restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
 Software, and to permit persons to whom the Software is furnished to do so,
 subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies
 or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <iostream>
#include <string>
#include <vector>
/* #include "rng.h" */
#include "stringmaniplib.h"
#include "chunkexpressionlib.h"
using namespace std;

char cxl_element_delimiter = 127;
char cxl_item_delimiter = ',';
char cxl_list_delimiter = 8;
vector<char> cxl_delimiter_stack;
union cxl_x_hasher
{
	char char_value[4];
	long int hash_value;
};

int cxl_count_chunks(const string & text, const string & chunktype)
{
	int c,l;
	cxl_x_hasher h;
	if (text == (string)"") { return 0; }
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	/*return h.hash_value;*/

	switch(h.hash_value) {
	case 'char':
		return (signed int)text.size();
		break;
	case 'word':
		return cxl_word_count(text);
		break;
	case 'item':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_item_delimiter) { c++; }
		}
		return c;
		break;
	case 'list':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_list_delimiter) { c++; }
		}
		return c;
		break;
	case 'elem':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_element_delimiter) { c++; }
		}
		return c;
		break;
	case 'line':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if ((text[l] == 13) || (text[l] == 10)) { c++; }
		}
		return c;
		break;
	case 'para':
		return cxl_paragraph_count(text);
		break;
	case 'sent':
		return cxl_sentence_count(text);
		break;
	case 'byte':
		return (signed int)text.size();
		break;
	case 'shor':
		return cxl_intdivceil((signed int)text.size(),2);
		break;
	case 'long':
		return cxl_intdivceil((signed int)text.size(),4);
		break;
	}
	return 0;
}

int cxl_x_fixStart(int startNum, int c, int h)
{
	if ( startNum == (int)0x7FFFFFFF ) /* middle, round up */
		{ return (c / 2) + 1; }
	else if ( startNum == (int)0x80000000 ) /* middle, round down */
		{ return (c / 2) + (c % 2); }
	else if ( startNum == (int)0x80000001 ) /* any */
		{ return 1 + (rand() % c); }
	else if ( (startNum < 0) && ((h == 'byte') || (h == 'shor') || (h == 'long')) )
		{ return c+startNum; }
	else if (startNum < 0)
		{ return c+startNum+1; }
	else
		{ return startNum; }
}

int cxl_x_fixEnd(int startNum, int endNum, int c, int h)
{
	if ( endNum == (int)0x7FFFFFFF ) /* middle, round up */
		{ return (c / 2) + 1; }
	else if ( endNum == (int)0x80000000 ) /* middle, round down */
		{ return (c / 2) + (c % 2); }
	else if ( endNum == (int)0x80000001 ) /* any */
		{ return startNum; } /* we must return the same thing in order to get *one* random chunk instead of a random range */
	else if ( (endNum < 0) && ((h == 'byte') || (h == 'shor') || (h == 'long')) )
		{ return cxl_max(cxl_min(c,c+endNum),startNum); }
	else if (endNum < 0)
		{ return cxl_max(cxl_min(c,c+endNum+1),startNum); }
	else
		{ return cxl_max(cxl_min(c,endNum),startNum); }
}

int cxl_chunk_start(const string & text, const string & chunktype, int startNum, int endNum)
{
	int l,c;
	int startNum2,endNum2;
	cxl_x_hasher h;
	if (text == (string)"") { return -1; }
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	switch(h.hash_value) {
	case 'char':
		return startNum2 - 1;
		break;
	case 'word':
		return cxl_word_start(text, startNum2);
		break;
	case 'item':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (c >= startNum2) { return l; }
			if (text[l] == cxl_item_delimiter) { c++; }
		}
		return (signed int)text.size();
		break;
	case 'list':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (c >= startNum2) { return l; }
			if (text[l] == cxl_list_delimiter) { c++; }
		}
		return (signed int)text.size();
		break;
	case 'elem':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (c >= startNum2) { return l; }
			if (text[l] == cxl_element_delimiter) { c++; }
		}
		return (signed int)text.size();
		break;
	case 'line':
		c = 1;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (c >= startNum2) { return l; }
			if ((text[l] == 13) || (text[l] == 10)) { c++; }
		}
		return (signed int)text.size();
		break;
	case 'para':
		return cxl_paragraph_start(text, startNum2);
		break;
	case 'sent':
		return cxl_sentence_start(text, startNum2);
		break;
	case 'byte':
		return startNum2;
		break;
	case 'shor':
		return startNum2*2;
		break;
	case 'long':
		return startNum2*4;
		break;
	}
	return -1;
}

int cxl_chunk_end(const string & text, const string & chunktype, int startNum, int endNum)
{
	int l,c;
	int startNum2,endNum2;
	cxl_x_hasher h;
	if (text == (string)"") { return -1; }
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	switch(h.hash_value) {
	case 'char':
		return endNum2 - 1;
		break;
	case 'word':
		return cxl_word_end(text, endNum2);
		break;
	case 'item':
		c = 0;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_item_delimiter)
			{
				c++;
				if (c >= endNum2) { return l-1; }
			}
		}
		return (signed int)text.size();
		break;
	case 'list':
		c = 0;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_list_delimiter)
			{
				c++;
				if (c >= endNum2) { return l-1; }
			}
		}
		return (signed int)text.size();
		break;
	case 'elem':
		c = 0;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if (text[l] == cxl_element_delimiter)
			{
				c++;
				if (c >= endNum2) { return l-1; }
			}
		}
		return (signed int)text.size();
		break;
	case 'line':
		c = 0;
		for (l=0; l<(signed int)text.size(); l++)
		{
			if ((text[l] == 13) || (text[l] == 10))
			{
				c++;
				if (c >= endNum2) { return l-1; }
			}
		}
		return (signed int)text.size();
		break;
	case 'para':
		return cxl_paragraph_end(text, endNum2);
		break;
	case 'sent':
		return cxl_sentence_end(text, endNum2);
		break;
	case 'byte':
		return endNum2;
		break;
	case 'shor':
		return (endNum2*2)+1;
		break;
	case 'long':
		return (endNum2*4)+3;
		break;
	}
	return -1;
}

string cxl_delete_chunk(const string & text, const string & chunktype, int startNum, int endNum)
{
	int c,startNum2,endNum2,startNum3,endNum3;
	string s;
	cxl_x_hasher h;
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	/* We need to handle any negative ranges before we pass them on to chunk_start/end, */
	/* since we increase the end of the range by 1. You'll see in a second. */
	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	startNum3 = cxl_chunk_start(text, chunktype, startNum2, startNum2);
	endNum3 = cxl_chunk_start(text, chunktype, endNum2+1, endNum2+1);
	/* No, you're not seeing things; that's the correct way to do it. */

	if ((startNum3) < 0 || (endNum3 < 0)) { return text; }
	s = left(text, startNum3) + mid(text, endNum3+1);
	return s;
}

string cxl_get_chunk(const string & text, const string & chunktype, int startNum, int endNum)
{
	int c,startNum2,endNum2,startNum3,endNum3;
	cxl_x_hasher h;
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	startNum3 = cxl_chunk_start(text, chunktype, startNum2, endNum2);
	endNum3 = cxl_chunk_end(text, chunktype, startNum2, endNum2);
	int l = cxl_max(1, endNum3-startNum3+1);
	string s;
	if ((startNum3) < 0 || (endNum3 < 0)) { return ""; }
	s = mid(text, startNum3+1, l);
	return s;
}

string cxl_put_into_chunk(const string & txt, const string & chunktype, int startNum, int endNum, const string & newtext)
{
	int c,l,startNum2,endNum2,startNum3,endNum3;
	string s;

	/* All this is necessary to properly handle putting into out-of-range items, lists, elements, or lines. */
	string text = txt;
	string nl = " ";
	cxl_x_hasher h;
	c = cxl_count_chunks(text, chunktype);
	nl[0] = 13; /* change to 10 if we want UNIX line endings */
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	if ( (c < startNum2) && ((h.hash_value == 'item') || (h.hash_value == 'list') || (h.hash_value == 'elem') || (h.hash_value == 'line')) )
	{
		c = startNum2-c;
		if (text == (string)"") { c--; }
		switch(h.hash_value) {
		case 'item':
			for (l=0; l<c; l++) { text += cxl_item_delimiter; }
			break;
		case 'list':
			for (l=0; l<c; l++) { text += cxl_list_delimiter; }
			break;
		case 'elem':
			for (l=0; l<c; l++) { text += cxl_element_delimiter; }
			break;
		case 'line':
			for (l=0; l<c; l++) { text += nl; }
			break;
		}
		c = cxl_count_chunks(text, chunktype);
	}

	startNum3 = cxl_chunk_start(text, chunktype, startNum2, endNum2);
	endNum3 = cxl_chunk_end(text, chunktype, startNum2, endNum2);
	l = cxl_max(1, endNum3-startNum3+1);
	if ((startNum3 < 0) || (endNum3 < 0)) { return newtext; }
	s = left(text, startNum3) + newtext + mid(text, endNum3+2);
	return s;
}

string cxl_put_before_chunk(const string & text, const string & chunktype, int startNum, int endNum, const string & newtext)
{
	int c,startNum2,endNum2;
	cxl_x_hasher h;
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	string t;
	t = newtext + cxl_get_chunk(text, chunktype, startNum2, endNum2);
	return cxl_put_into_chunk(text, chunktype, startNum2, endNum2, t);
}

string cxl_put_after_chunk(const string & text, const string & chunktype, int startNum, int endNum, const string & newtext)
{
	int c,startNum2,endNum2;
	cxl_x_hasher h;
	c = cxl_count_chunks(text, chunktype);
	h.char_value[0] = cxl_tolower(chunktype[0]);
	h.char_value[1] = cxl_tolower(chunktype[1]);
	h.char_value[2] = cxl_tolower(chunktype[2]);
	h.char_value[3] = cxl_tolower(chunktype[3]);

	startNum2 = cxl_x_fixStart( startNum, c, h.hash_value );
	endNum2 = cxl_x_fixEnd( startNum2, endNum, c, h.hash_value );

	string t;
	t = cxl_get_chunk(text, chunktype, startNum2, endNum2) + newtext;
	return cxl_put_into_chunk(text, chunktype, startNum2, endNum2, t);
}

int cxl_paragraph_count(const string & s)
{
	int n,r;
	string text = s;
	char cr = 13;
	char lf = 10;
	if (trim(text) == (string)"") { return 0; }
	n = 0;
	r = 0;
	while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
	while (text[n] != 0)
	{
		while ( (text[n] != cr) && (text[n] != lf) && (text[n] != 0) ) { n++; }
		while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
		r++;
	}
	return r;
}

int cxl_paragraph_start(const string & text, int theNum)
{
	int i,n;
	char cr = 13;
	char lf = 10;
	n = 0;
	while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (text[n] != cr) && (text[n] != lf) && (text[n] != 0) ) { n++; }
			while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
		}
	}
	return n;
}

int cxl_paragraph_end(const string & text, int theNum)
{
	int i,n;
	char cr = 13;
	char lf = 10;
	n = 0;
	while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (text[n] != cr) && (text[n] != lf) && (text[n] != 0) ) { n++; }
			while ( ((text[n] == cr) || (text[n] == lf)) && (text[n] != 0) ) { n++; }
		}
	}
	while ( (text[n] != cr) && (text[n] != lf) && (text[n] != 0) ) { n++; }
	return n-1;
}

int cxl_sentence_count(const string & s)
{
	int n,r;
	bool b;
	string text = s;
	if (trim(text) == (string)"") { return 0; }
	n = 0;
	r = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	while (text[n] != 0)
	{
		while ( (! cxl_x_sentence_ender(text[n])) && (text[n] != 0) ) { n++; }
		b = true;
		while ( cxl_x_skippable_a(text[n],b) && (text[n] != 0) )
		{
			if ( cxl_x_whitespace(text[n]) ) { b = false; }
			n++;
		}
		r++;
	}
	return r;
}

int cxl_sentence_start(const string & text, int theNum)
{
	int i,n;
	bool b;
	n = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (! cxl_x_sentence_ender(text[n])) && (text[n] != 0) ) { n++; }
			b = true;
			while ( cxl_x_skippable_a(text[n],b) && (text[n] != 0) )
			{
				if ( cxl_x_whitespace(text[n]) ) { b = false; }
				n++;
			}
		}
	}
	return n;
}

int cxl_sentence_end(const string & text, int theNum)
{
	int i,n;
	bool b;
	n = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (! cxl_x_sentence_ender(text[n])) && (text[n] != 0) ) { n++; }
			b = true;
			while ( cxl_x_skippable_a(text[n],b) && (text[n] != 0) )
			{
				if ( cxl_x_whitespace(text[n]) ) { b = false; }
				n++;
			}
		}
	}
	while ( (! cxl_x_sentence_ender(text[n])) && (text[n] != 0) ) { n++; }
	while ( cxl_x_skippable_b(text[n]) && (text[n] != 0) ) { n++; }
	return n-1;
}

int cxl_word_count(const string & s)
{
	int n,r;
	string text = s;
	if (trim(text) == (string)"") { return 0; }
	n = 0;
	r = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	while (text[n] != 0)
	{
		while ( (! cxl_x_whitespace(text[n])) && (text[n] != 0) ) { n++; }
		while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
		r++;
	}
	return r;
}

int cxl_word_start(const string & text, int theNum)
{
	int i,n;
	n = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (! cxl_x_whitespace(text[n])) && (text[n] != 0) ) { n++; }
			while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
		}
	}
	return n;
}

int cxl_word_end(const string & text, int theNum)
{
	int i,n;
	n = 0;
	while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
	if (theNum > 1)
	{
		for (i=2; i<=theNum; i++)
		{
			while ( (! cxl_x_whitespace(text[n])) && (text[n] != 0) ) { n++; }
			while ( cxl_x_whitespace(text[n]) && (text[n] != 0) ) { n++; }
		}
	}
	while ( (! cxl_x_whitespace(text[n])) && (text[n] != 0) ) { n++; }
	return n-1;
}

bool cxl_x_quote_mark(unsigned char c)
{
	return ( (c==34) || (c==39) || (c==210) || (c==211) || (c==212) || (c==213) || (c==199) || (c==200) );
}

bool cxl_x_sentence_ender(unsigned char c)
{
	return ( (c=='.') || (c=='!') || (c=='?') );
}

bool cxl_x_whitespace(unsigned char c)
{
	return ( (c==13) || (c==10) || (c==32) || (c==9) || (c==0) );
}

bool cxl_x_skippable_a(unsigned char c, bool b)
{
	return ( cxl_x_whitespace(c) || cxl_x_sentence_ender(c) || (b && cxl_x_quote_mark(c)) || (c==')') );
}

bool cxl_x_skippable_b(unsigned char c)
{
	return ( cxl_x_sentence_ender(c) || cxl_x_quote_mark(c) || (c=='(') );
}

void cxl_reset_delimiters(void)
{
	cxl_element_delimiter = 127;
	cxl_item_delimiter = 34;
	cxl_list_delimiter = 8;
}

void cxl_push_item_delimiter(void)
{
	cxl_delimiter_stack.push_back(cxl_item_delimiter);
}

void cxl_push_element_delimiter(void)
{
	cxl_delimiter_stack.push_back(cxl_element_delimiter);
}

void cxl_push_list_delimiter(void)
{
	cxl_delimiter_stack.push_back(cxl_list_delimiter);
}

void cxl_pop_item_delimiter(void)
{
	if (cxl_delimiter_stack.empty()) { cxl_item_delimiter = 34; }
	else
	{
		cxl_item_delimiter = *(cxl_delimiter_stack.end() - 1);
		cxl_delimiter_stack.pop_back();
	}
}

void cxl_pop_element_delimiter(void)
{
	if (cxl_delimiter_stack.empty()) { cxl_element_delimiter = 127; }
	else
	{
		cxl_element_delimiter = *(cxl_delimiter_stack.end() - 1);
		cxl_delimiter_stack.pop_back();
	}
}

void cxl_pop_list_delimiter(void)
{
	if (cxl_delimiter_stack.empty()) { cxl_list_delimiter = 8; }
	else
	{
		cxl_list_delimiter = *(cxl_delimiter_stack.end() - 1);
		cxl_delimiter_stack.pop_back();
	}
}



chunk_reference::chunk_reference(void)
{
	type = "char";
	start = 0;
	end = 0;
}

chunk_reference::chunk_reference(const string & t, int s, int e)
{
	type = t;
	start = s;
	end = e;
}

chunk_reference::chunk_reference(const chunk_reference & cr)
{
	type = cr.type;
	start = cr.start;
	end = cr.end;
}

const chunk_reference & chunk_reference::operator = (const chunk_reference & cr)
{
	type = cr.type;
	start = cr.start;
	end = cr.end;
	return *this;
}



vector<chunk_reference> cxl_cexpr_partial_parse(const string & s)
{
	vector<chunk_reference> v;
	chunk_reference cr;
	string t,x;
	int i,m;
	t=replaceall((string)" "+s+(string)" ",(string)" of ",(string)";");
	t=replaceall((string)" "+t+(string)" ",(string)" to ",(string)",");
	t=trim(t);
	t=replaceall(t,(string)" ",(string)",");
	m=countfields(t, ";");
	for (i=1; i<=m; i++)
	{
		x = nthfield(t, ";", i);
		cr.type = nthfield(x, ",", 1);

		if (trim(nthfield(x, ",", 2)) == "any")
			{ cr.start = 0x80000001; }
		else if (trim(nthfield(x, ",", 2)) == "mid")
			{ cr.start = 0x7FFFFFFF; }
		else
			{ cr.start = (int)val(nthfield(x, ",", 2)); }

		if (trim(nthfield(x, ",", 3)) == "")
		{
			cr.end = cr.start;
		} else {
			cr.end = (int)val(nthfield(x, ",", 3));
		}
		v.push_back(cr);
	}
	return v;
}

int cxl_cexpr_chunk_start(const string & text, const vector<chunk_reference> & expr)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	int a,b;
	string t;

	/* General */
	if (expr.empty()) { return 0; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	a = cxl_chunk_start(text,ct,cs,ce);
	if (rst.empty())
	{
		return a;
	} else {
		t = cxl_get_chunk(text,ct,cs,ce);
		b = cxl_cexpr_chunk_start(t, rst);
		if ( (a<0) || (b<0) || (a>=(signed int)text.size()) || (b>=(signed int)text.size()) ) { return 0; }
		else { return a+b; }
	}
}

int cxl_cexpr_chunk_end(const string & text, const vector<chunk_reference> & expr)
{
	string t = cxl_cexpr_get_chunk(text, expr);
	int i = cxl_cexpr_chunk_start(text, expr);
	return i+t.size()-1;
}

string cxl_cexpr_delete_chunk(const string & text, const vector<chunk_reference> & expr)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	string a;

	/* General */
	if (expr.empty()) { return text; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	if (rst.empty())
	{
		return cxl_delete_chunk(text, ct, cs, ce);
	} else {
		a = cxl_get_chunk(text,ct,cs,ce);
		a = cxl_cexpr_delete_chunk(a, rst);
		a = cxl_put_into_chunk(text, ct, cs, ce, a);
		return a;
	}
}

string cxl_cexpr_get_chunk(const string & text, const vector<chunk_reference> & expr)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	string a;

	/* General */
	if (expr.empty()) { return text; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	a = cxl_get_chunk(text,ct,cs,ce);
	if (rst.empty())
	{
		return a;
	} else {
		return cxl_cexpr_get_chunk(a, rst);
	}
}

string cxl_cexpr_put_into_chunk(const string & text, const vector<chunk_reference> & expr, const string & newtext)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	string a;

	/* General */
	if (expr.empty()) { return text; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	if (rst.empty())
	{
		return cxl_put_into_chunk(text, ct, cs, ce, newtext);
	} else {
		a = cxl_get_chunk(text,ct,cs,ce);
		a = cxl_cexpr_put_into_chunk(a, rst, newtext);
		a = cxl_put_into_chunk(text, ct, cs, ce, a);
		return a;
	}
}

string cxl_cexpr_put_before_chunk(const string & text, const vector<chunk_reference> & expr, const string & newtext)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	string a;

	/* General */
	if (expr.empty()) { return text; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	if (rst.empty())
	{
		return cxl_put_before_chunk(text, ct, cs, ce, newtext);
	} else {
		a = cxl_get_chunk(text,ct,cs,ce);
		a = cxl_cexpr_put_before_chunk(a, rst, newtext);
		a = cxl_put_into_chunk(text, ct, cs, ce, a);
		return a;
	}
}

string cxl_cexpr_put_after_chunk(const string & text, const vector<chunk_reference> & expr, const string & newtext)
{
	/* Assignments General to the Recursive Routines */
	vector<chunk_reference> rst;
	chunk_reference lst;
	string ct;
	int cs,ce;

	/* Assignments Specific to this Routine */
	string a;

	/* General */
	if (expr.empty()) { return text; }
	rst = expr;
	lst = rst.back();
	rst.pop_back();
	ct = lst.type;
	cs = lst.start;
	ce = lst.end;
	if (ce == 0) { ce = cs; }

	/* Specific */
	if (rst.empty())
	{
		return cxl_put_after_chunk(text, ct, cs, ce, newtext);
	} else {
		a = cxl_get_chunk(text,ct,cs,ce);
		a = cxl_cexpr_put_after_chunk(a, rst, newtext);
		a = cxl_put_into_chunk(text, ct, cs, ce, a);
		return a;
	}
}

